function [BcP, AcP, K0Q_cP, K1Q_cP, rho0_cP, rho1_cP, K0Q_X, K1Q_X, AX_all, BX_all, Sigma_X, alpha0_cP, alpha1_cP, alpha0_X_all, alpha1_X_all, m1] = jszLoadings_rho0cP_r1(W, K1Q_X, rho0_cP, Sigma_cP, rho0_X_r, rho1_X_r, mats, mats_r, dt, Sigma_X)
%
% This gives a slight variant of the JSZ normalization.
% 
% Adapted from JSZ code for RDP Hambur and Finlay (2018)
%
% There is one "intercept" parameter governing the risk-neutral
% distribution.  This can be the long run mean of the short rate under Q
% (assuming stationarity) or the drift of the most persistent factor.
%
% Here we parameterize the intercept parameter through the relationship:
%  r_t = rho0_cP + rho1_cP.Xt
% 
% Given a fixed set of eigenvalues for K1Q, there is a one-to-one
% mapping between rho0_cP and kinf.
%
%
% Inputs:
%   mats       : 1*J,      maturities in years
%   dt         : scalar,   length of period in years
%   W          : N*J,      vector of portfolio weights to fit without error.
%   K1Q_X      : N*N
%   rho0_cP    : scalar,   the short rate will be rt = rho0_cP + rho1_cP.cP_t
%   rho0_X_r    : scalar,   the inflation short rate will be pi_t = rho0_X_r + rho1_X_r.X_t
%   rho0_X_r    : N*1,   the inflation short rate will be pi_t = rho0_X_r + rho1_X_r.X_t
%   Sigma_cP, Sigma_X : N*N  covariance of innovations. PROVIDE ONE OR THE OTHER
%
% Returns:
%   AcP : 1*J
%   BcP : N*J
%   AX  : 1*J
%   BX  : N*J
%   Sigma_X : N*N
%   AcP = alpha0_cP*kinf + alpha1_cP
%   AX  = alpha0_X*kinf  + alpha1_X
%
%
% This function:
% 1. Compute the loadings for the normalized model:
%   X(t+1) - X(t)   = [kinfQ;0] + K1Q_X*X(t)  + eps_X(t+1),   cov(eps_X(t+1)) = Sigma_X
%     and r(t) = 1.X(t)  
%     where r(t) is the annualized short rate, (i.e. price of 1-period zero coupon bond at time t is exp(-r(t)*dt))
%    If Sigma_X is not provided, it is solved for so that Sigma_cP (below) is matched.
%    yt = AX' + BX'*Xt
%
% 2. For cPt = W*yt and the model above for Xt, find AcP, BcP so that
%    yt = AcP' + BcP'*cPt
%
%

mats_all=[mats, mats_r];
J = length(mats_all);
N = size(K1Q_X,1);
rho0d = 0; %Nomalised short-rate coeff
rho1d = ones(N,1);
mats_periods = round(mats/dt);
mats_periods_r = round(mats_r/dt);
M = max(mats_periods);

[K1Q_X, isTypicalDiagonal, m1] = jszAdjustK1QX(K1Q_X);
K0Q_X = zeros(N,1);
K0Q_X(m1) = 1; 



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%
% In this section, because we don't have K0Q_X, we do a few tricks to get
% to it
% First get A and B matrices assuming Sigma_X=0. BX will be
% correct as it relies on rho1_X and K1Q_X  (see iteractive equation in appendix)

% Next use B matrix to calculate Sigma X, using the following derivation 
% yt = AX' + BX'*Xt
% where AX = alpha0_X*kinf + alpha1_X (second term is convexity adjustment)
% yt is J*1
% AX is 1*J
% BX is N*J
% Xt is N*1
%
% cPt = W*yt  (cPt N*1, W is N*J)
%     = W*AX' + W*BX'*Xt
%     = WAXp + WBXp*Xt
%
% Substituting:
% yt = AX' + BX'*(WBXp\(cPt - WAXp))
%    = (I - BX'*(WBXp\WAXp))*AX' + BX'*WBXp\cPt
%    = AcP' + BcP'*cPt
% where AcP = AX*(I - BX'*(WBXp\WAXp))'
%       BcP = (WBXp)'\BX
%
% Sigma_cP = W*BX'*Sigma_X*(W*BX')'
% Sigma_X = (W*BX')\Sigma_cP/(W*BX')'
%


% If K1d isn't diagonal, we should use the Recurrence solver:.
% Since we are setting covariance to zero and K0Q_X = [1;0;0..;0], the "A"
% loadings will be the loading on kinf (which we set to 1)
if isTypicalDiagonal
    [BX, alpha0_X] = gaussianDiscreteYieldLoadingsDiagonal(mats_periods, K0Q_X, diag(K1Q_X), zeros(N,N), rho0d*dt, rho1d*dt, dt); % N*J
else
    [BX, alpha0_X] = gaussianDiscreteYieldLoadingsRecurrence(mats_periods, K0Q_X, K1Q_X, zeros(N,N), rho0d*dt, rho1d*dt, dt); % N*J
end

% Need set rho1_X_r = 0 here to isolate B matrix - again see iterative equations in appendix
[BX_r, alpha0_X_r] = gaussianDiscreteYieldLoadingsRecurrence_real(mats_periods_r, K0Q_X, K1Q_X, zeros(N,N), 0, rho1_X_r*dt, rho0d*dt, rho1d*dt, dt); 



BX_all=[BX, BX_r];
WBXp = W*BX_all.'; % N*N
if nargin<10 || isempty(Sigma_X)
    Sigma_X = (W*BX_all')\Sigma_cP/(BX_all*W');
end

alpha0_X_all= [alpha0_X, alpha0_X_r];
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Now with Sigma_X in hand, compute loadings for AX for the full loading
if isTypicalDiagonal
    [BX, AX1] = gaussianDiscreteYieldLoadingsDiagonal(mats_periods, K0Q_X, diag(K1Q_X), Sigma_X, rho0d*dt, rho1d*dt, dt);
else
    [BX, AX1] = gaussianDiscreteYieldLoadingsRecurrence(mats_periods, K0Q_X, K1Q_X, Sigma_X, rho0d*dt, rho1d*dt, dt);
end


[BX_r, AX1_r] = gaussianDiscreteYieldLoadingsRecurrence_real(mats_periods_r, K0Q_X, K1Q_X, Sigma_X, rho0_X_r*dt, rho1_X_r*dt, rho0d*dt, rho1d*dt, dt); % N*J

% Finally, use the difference between the two to split A into the scale
% invariant convexxity adjustment, alpha1_x, and the coeffcient on kinf, alpha0_X
%
% AX1 gives the intercept with K0Q_X all zeros except 1 in the m1-th entry.
% So AX1 = alpha0_X*kinf + alpha1_X which alpha1_X = AX1 - alpha0_X (as
% treated kinf as 1) is the convexity adjustments and rho_x_r term
% The first 'intercept' terms scales with the kinf (so K0Q_X), the other terms do not
% (not affected by level, just the sigma see Appendix)
alpha1_X = AX1 - alpha0_X;
alpha1_X_r = AX1_r - alpha0_X_r;

alpha1_X_all = [alpha1_X, alpha1_X_r];

BX_all=[BX, BX_r];
WBXp = W*BX_all.'; % N*N

% Need to find what kinf should be to get the desired rho0_cP:
% rt = 1'*Xt 
% cPt = W*AX' + W*BX'*Xt
% cPt = (W*alpha0_X')*kinf + (W*alpha1_X') + (W*BX')*Xt
% rt = 1'*(W*BX')^(-1)*[cPt -(W*alpha0_X')*kinf - (W*alpha1_X')]
% --> rho0_cP = -1'*(W*BX')^(-1)*(W*alpha0_X')*kinf - 1'*(W*BX')^(-1)*(W*alpha1_X')
a0 = ones(1,N)*(WBXp\(W*alpha0_X_all')); %(term on kinf)
a1 = ones(1,N)*(WBXp\(W*alpha1_X_all')); %intercept term 
kinf = -(rho0_cP + a1)/a0;
K0Q_X(m1) = kinf; 

AX_all = alpha0_X_all*kinf + alpha1_X_all;
% AcP = alpha0_cP*kinf + alpha1_cP
alpha0_cP = ((eye(J) - (BX_all'/(W*BX_all'))*W)*alpha0_X_all')';
alpha1_cP = ((eye(J) - (BX_all'/(W*BX_all'))*W)*alpha1_X_all')';
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Finally, rotate the model to obtain the AcP, BcP loadings.
% (See above for calculation)
BcP = (W*BX_all.').'\BX_all;
AcP = AX_all*(eye(J) - BX_all'*((W*BX_all')\W))'; % 1*J
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Now compute the rotated model parameters:
WBXp = W*BX_all';
WAXp = W*AX_all';

K1Q_cP = WBXp*K1Q_X/WBXp;
K0Q_cP = WBXp*K0Q_X - K1Q_cP*WAXp;

rho0_cP =  - ones(1,N)*(WBXp\WAXp);
rho1_cP = (WBXp)'\ones(N,1);
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%