%% DensIndex
%
% This function computes density indices to analyse the changing
% relationships between core and periphery (accounting for changes in the
% core and periphery).
%
% =========================================================
% AUTHOR AND INSTITUTION INFO:
% Written by Anthony Brassil and Gabriela Nodari
% Reserve Bank of Australia
% This version: January 2018
% ==========================================================

function [Dens] = DensIndex(CPvecGlob,Act,adj_ALL)

% INPUT:
% CPvecGlob is a {1xT} cell array containing the core/periphery structure
% of each network in the sample.
% Act is a TxN matrix. Cell (t,i) contains the status of node i
% in network t. If any loans in network t, the bank is active and has a
% value of one. If never any loans, the node is never active and has a value
% of NaN. If not active in network t but active in some network s~=t, then
% (t,i) has a value of zero.
% adj_ALL is a {1xT} cell array containing the adjacency matrices of each
% network in the sample.

% OUTPUT:
% Dens is a Tx2 matrix - density index
%  - 1 is the core to periphery index, 2 is the periphery to core index
%  - index is re-based to a mean of 100

T = size(Act,1);
N = size(Act,2);
CPvec = zeros(N,T);
for t = 1:T
    CPvec(:,t) = CPvecGlob{1,t}';
end

Dens = zeros(T,2);
Dens(1,:) = 100;

for t=2:T
    % Retrieve adjacency matrices and active nodes for each network
    adj_mtx = adj_ALL{1,t};
    adj_mtxp = adj_ALL{1,t-1};
    nodes = Act(t,:);
    nodesp = Act(t-1,:);

    % Construct density index
    tmpPC = adj_mtx(CPvec(:,t)==0 & CPvec(:,t-1)==0 & nodes'==1 & nodesp'==1, CPvec(:,t)'==1 & CPvec(:,t-1)'==1);
    tmpPCp = adj_mtxp(CPvec(:,t)==0 & CPvec(:,t-1)==0 & nodes'==1 & nodesp'==1, CPvec(:,t)'==1 & CPvec(:,t-1)'==1);
    tmpCP = adj_mtx(CPvec(:,t)==1 & CPvec(:,t-1)==1, CPvec(:,t)'==0 & CPvec(:,t-1)'==0 & nodes==1 & nodesp==1);
    tmpCPp = adj_mtxp(CPvec(:,t)==1 & CPvec(:,t-1)==1, CPvec(:,t)'==0 & CPvec(:,t-1)'==0 & nodes==1 & nodesp==1);
    if sum(tmpCP(:))>0 && sum(tmpCPp(:))>0
        Dens(t,1) = (sum(tmpCP(:))/sum(tmpCPp(:)))*Dens(t-1,1);
    else
        Dens(t,1) = Dens(t-1,1);
    end
    if sum(tmpPC(:))>0 && sum(tmpPCp(:))>0
        Dens(t,2) = (sum(tmpPC(:))/sum(tmpPCp(:)))*Dens(t-1,2);
    else
        Dens(t,2) = Dens(t-1,2);
    end
end

% Re-base density index
Dens(:,1) = (Dens(:,1)./mean(Dens(:,1))).*100;
Dens(:,2) = (Dens(:,2)./mean(Dens(:,2))).*100;

end

