%% plot_figures_6_7

% =========================================================
% AUTHORS AND INSTITUTION INFO:
% Code written by Anthony Brassil and Gabriela Nodari
% Reserve Bank of Australia
% This version: January 2018
% ==========================================================

% This code runs numerical simulations to determine the accuracy of the
% estimators. Simulations are run for multiple core sizes.

% IMPORTANT:
% As is explained in Appendix D, these simulations are constructed to
% satisfy certain restrictions. The way these restrictions are imposed
% depends on the simulation parameterisation. Therefore, deviating from our
% parameterisations may cause this code to produce incorrect simulations 
% (this is also discussed in Appendix D). 
% When deviating from our parameterisations, we suggest starting from 
% Equation D1 in Appendix D and then deriving versions of equations D2-D5
% that are suitable for your parameterisation.
% The code below will produce error messages when the code is not suitable
% for a given parameterisation, but it will not explain how to adjust the 
% code to suit the new parameterisation.

%% Initialisation
clear all
clc
close all

%% Parameters
Smpl = 10000; % Number of random draws for each parameterisation
N = 40; % Number of nodes (calibrated to number of active banks in the data)
VersSt = 1; % Initial estimator to test (based on errorfn.m)
Vers = 4; % Final estimator to test (based on errorfn.m)
Iter = 20; % Number of random starting points for greedy (steepest descent) algorithm
DT = 0.25; % Density of the entire network (calibrate to average density of active banks in the data)
CN = [0; 0.05; 0.1; 0.15; 0.2; 0.25; 0.3; 0.35; 0.4; 0.45]; % Proportion of N that is in the core
vnames = cellstr(['Corr';'ML  ';'DB  ';'CvP ']); % Abbreviations for the four estimators
ci = 0.05; % Set to 0.05 to construct the 95th percentile
flag_dC = 1; % Set to 1 to restrict dC=1 (the top panels of Figures 6 and 7), set to 0 to replicate the bottom panels of Figures 6 and 7;

NumEdg = round(N*(N-1)*DT); % Edges in this network
if flag_dC == 1
    filename1 = ['Simulation_tmp_dC1_',num2str(Smpl),'.mat'];
    filename2 = ['Simulation_dC1_',num2str(Smpl),'.mat'];
else
    filename1 = ['Simulation_tmp_',num2str(Smpl),'.mat'];
    filename2 = ['Simulation_',num2str(Smpl),'.mat'];
end

%% Construct simulated networks and initialise arrays and loops

if exist(filename1,'file')==2
    load(filename1) % So the code can be re-started from the last saved point if it gets interrupted.
else
    loans = zeros(N,N,size(CN,1),Smpl,'uint64');

    for i = 1:size(CN,1)
        C = round(CN(i)*N);
        for s = 1:Smpl
            % STEP 1 - draw block densities
            % See Appendix D for an explanation of this step.
            
            % Draw r
            if flag_dC == 1
                r = 1;
            else
                r = rand;
            end
            
            % Error checking
            if (((N-C)*(N-C-1))+((r-1)*C*(C-1))) <= 0
                disp('Error: (((N-C)*(N-C-1))+((r-1)*C*(C-1)))>0 must hold')
            end
            if ((DT*N*(N-1))-(C*(C-1)))<=0
                disp('Error: ((DT*N*(N-1))-(C*(C-1)))>0 must hold')
            end
            if (((r-1)*DT*N*(N-1))+((N-C)*(N-C-1)))<=0
                disp('Error: (((r-1)*DT*N*(N-1))+((N-C)*(N-C-1)))>0 must hold')
            end
            
            % Max off-diagonal density (Part 1)
            if C>0
                max_oT1 = ((DT*N*(N-1))-(C*(C-1)))/(2*C*(N-C));
            else
                max_oT1 = DT;
            end
            
            % Min off-diagonal density
            if C>0
                min_oT = (((DT*N*(N-1))-(C*(C-1))))/((((N-C)*(N-C-1)))+(2*C*(N-C))+((r-1)*C*(C-1)));
            else
                min_oT = DT;
            end
            
            % Max off-diagonal density (Part 2)
            if C>0
                minmax_oT2 = (((r-1)*DT*N*(N-1))+((N-C)*(N-C-1)))/(((N-C)*(N-C-1))+((r-1)*C*(C-1))+(2*(r-1)*C*(N-C)));
            else
                minmax_oT2 = DT;
            end
            
            % Draw off-diagonal density
            if (((N-C)*(N-C-1))+((r-1)*C*(C-1))+(2*(r-1)*C*(N-C))) > 0
                dens_oT = (rand*(min(max_oT1,minmax_oT2)-min_oT)) + min_oT;
                if min(max_oT1,minmax_oT2)<min_oT
                    disp('Error: upper bound is less than lower bound')
                end
            else
                dens_oT = (rand*(max_oT1-min_oT)) + min_oT;
            end

            % Construct periphery density
            if C>0
                dens_pT = (((DT*N*(N-1))-(C*(C-1)))-(2*dens_oT*C*(N-C)))/(((N-C)*(N-C-1))+((r-1)*C*(C-1)));
            else
                dens_pT = DT;
            end            
            
            % Construct core density
            dens_cT = (r*dens_pT) + 1 - dens_pT;
            
            % Check that restrictions hold
            if (dens_cT<dens_oT) || (dens_oT<dens_pT) || ((1-dens_cT)>dens_pT)
                disp('Error: restrictions do not hold')
            end
            
            % STEP 2
            % Construct blocks using Erdos-Renyi model
            if C>0
                loans(1:C,1:C,i,s) = mrandi(C,C,dens_cT);
                for k = 1:C
                    loans(k,k,i,s) = 0;
                end                
            end
            if C<N
                loans(C+1:N,C+1:N,i,s) = mrandi(N-C,N-C,dens_pT);
                for k = C+1:N
                    loans(k,k,i,s) = 0;
                end                 
            end
            if C>0 && C<N
                loans(1:C,C+1:N,i,s) = mrandi(C,N-C,dens_oT);
                loans(C+1:N,1:C,i,s) = mrandi(N-C,C,dens_oT);
            end

            % STEP 3
            % Ensure true off-diagonal blocks are row/column regular
            if C>0 && C<N
                for h = 1:C
                    if sum(loans(C+1:N,h,i,s))==0
                        loans(C+randi(N-C),h,i,s) = 1;
                    end
                    if sum(loans(h,C+1:N,i,s))==0
                        loans(h,C+randi(N-C),i,s) = 1;
                    end
                end
            end
        end
    end
    clear i k C s dens_pT dens_cT dens_oT h

    % Report average densities of simulated draws
    Dens = zeros(size(CN,1),1);
    Dens_o = zeros(size(CN,1),1);
    Dens_p = zeros(size(CN,1),1);
    Dens_c = zeros(size(CN,1),1);
    for i = 1:size(CN,1)
        C = round(CN(i)*N);
        Dens(i) = sum((sum(sum(loans(:,:,i,:),1),2)./((N^2)-N)),4)/Smpl;
        Dens_c(i) = sum((sum(sum(loans(1:C,1:C,i,:),1),2)./(C*(C-1))),4)/Smpl;
        Dens_o(i) = sum((sum(sum(loans(1:C,C+1:N,i,:),1),2)./(C*(N-C))),4)/Smpl;
        Dens_o(i) = Dens_o(i) + sum((sum(sum(loans(C+1:N,1:C,i,:),1),2)./(C*(N-C))),4)/Smpl;
        Dens_o(i) = Dens_o(i)/2;
        Dens_p(i) = sum((sum(sum(loans(C+1:N,C+1:N,i,:),1),2)./((N-C)*(N-C-1))),4)/Smpl;
        str = ['Dens: ',num2str(Dens(i)),' Core: ',num2str(Dens_c(i)),' Off: ',num2str(Dens_o(i)),' Periph: ',num2str(Dens_p(i))];
        disp(str)        
    end
    clear i C

    CPvec = zeros(N,size(CN,1),Smpl,Vers);
    NumOpt = zeros(size(CN,1),Smpl,Vers);
    MinErr = zeros(size(CN,1),Smpl,Vers);
    SameVec = zeros(size(CN,1),Smpl,Vers);

    vtemp = VersSt;
    itemp = 1;
    save(filename1,'-v7.3')
end

%% Identify core/periphery from each simulated network
% These temporary variables change with each iteration and are saved. This
% creates a restart point to be used if the code is interrupted.
vstart = vtemp;
istart = itemp;

for v = vstart:Vers
    for i = istart:size(CN,1)
        for s = 1:Smpl
            [CPvec(:,i,s,v),NumOpt(i,s,v),MinErr(i,s,v),SameVec(i,s,v),~] = IntOptGlob(loans(:,:,i,s),ones(1,N),v,Iter);
            str = ['S: ',num2str(s),'/',num2str(Smpl),' CN: ',num2str(i),'/',num2str(size(CN,1)),' V: ',num2str(v),'/',num2str(Vers)];
            disp(str)
        end
        itemp = i;
        save(filename1,'-v7.3')
    end
    istart = 1;
    vtemp = v;
end

%% Define errors relative to the core
% Each incorrect identification is an error. For a maximum error of N.

Err = zeros(size(CN,1),Smpl,Vers);
Ideal = zeros(N,size(CN,1));
for i = 1:size(CN,1)
    C = round(CN(i)*N);
    if C>0
        Ideal(1:C,i) = 1;
    end
end
clear C

for i = 1:size(CN,1)
    for s = 1:Smpl
        for v = VersSt:Vers
            Err(i,s,v) = sum(abs(CPvec(:,i,s,v)-Ideal(:,i)));
        end
    end
end

AvgErr(:,:) = mean(Err,2); % [size(CN,1),Vers]
AvgNumCore(:,:) = mean(sum(CPvec,1),3); % [size(CN,1),Vers]

ErrLB = zeros(size(CN,1),Vers);
ErrUB = zeros(size(CN,1),Vers);
for v = VersSt:Vers
    for i = 1:size(CN,1)
        tmp(:,1) = sort(Err(i,:,v),2)';
        ErrLB(i,v) = tmp(round(Smpl*ci));
        ErrUB(i,v) = tmp(round(Smpl*(1-ci)));
        clear tmp
    end
end

%Plot Figure 6
figure
hold all
for v = VersSt:Vers
    plot(sum(Ideal,1)',AvgErr(:,v));
end
vnamesUB = vnames;
for v = VersSt:Vers
    plot(sum(Ideal,1)',ErrUB(:,v),'--');
    vnamesUB(v,1) = strcat(vnames(v,1),' UB');
end
legend([vnames(VersSt:Vers); vnamesUB(VersSt:Vers)]);
title('Average Error');
hold off

% Plot Figure 7
figure
hold all
for v = VersSt:Vers
    plot(sum(Ideal,1)',AvgNumCore(:,v));
end
plot(sum(Ideal,1)',sum(Ideal,1)');
legend(vnames(VersSt:Vers),'True');
title('Average Core Size');
hold off

%% Save output
save(filename2,'-v7.3')
delete(filename1)