%% getNodes
%
% This function classifies banks as active/inactive. For any period, 
% a bank is active if it has at least one transaction during that period.
%
% =========================================================
% AUTHOR AND INSTITUTION INFO:
% Written by Anthony Brassil and Gabriela Nodari
% Reserve Bank of Australia
% This version: January 2018
% ==========================================================

function [banks_status_mtx,tradingdays]=getNodes(loans_all,frequency)

% Define internal functions
rows = @(x) size(x,1);
cols = @(x) size(x,2);

% Get identification numbers for all banks in the dataset. Notice that some
% IDs may not be used, e.g. ID 44 may not be assigned to any bank. We deal
% with this issue by creating columns of 'NaNs'.

banks_idx = min(min(loans_all(:,2:3))):1:max(max(loans_all(:,2:3)));

% Get a list of all trading days
tradingdays = unique(loans_all(:,1));
% Find first and last year of sample
yearstart = year(tradingdays(1));
yearend = year(tradingdays(end));

%% Create a binary matrix with banks' status: 0 = inactive, 1 = active

if frequency == 1 % (Quarterly)
    
    quarterstart = quarter(tradingdays(1));
    quarterend = quarter(tradingdays(end));
    % Number of whole quarters between dates
    qtot = ((yearend - yearstart - 1)*4) + (4-quarterstart+1) + (quarterend) ;
    % Create quarterly matrix of banks' status: active/inactive
    banks_status_mtx = zeros(qtot,cols(banks_idx));
    
    i=1;
    while yearstart <= yearend && i<=rows(banks_status_mtx)
        
        dd1=datenum(yearstart,1,1);
        dd2=datenum(yearstart,3,31);
        tlower = find(loans_all(:,1)>=dd1,1,'first');
        tupper = find(loans_all(:,1)<=dd2,1,'last');
        for j = 1:cols(banks_idx)
            if isempty(find(loans_all(tlower:tupper,2:3)==j,1))==0
                banks_status_mtx(i,j)=1; % this only says if a bank had a transaction within a given quarter
            end
        end
        
        % Restarts if the first quarter in the sample is not quarter one
        if i==1 && quarterstart~=1
            i=1;
        else
            i=i+1;
        end
        
        dd1=datenum(yearstart,4,1);
        dd2=datenum(yearstart,6,30);
        tlower = find(loans_all(:,1)>=dd1,1,'first');
        tupper = find(loans_all(:,1)<=dd2,1,'last');
        for j = 1:cols(banks_idx)
            if isempty(find(loans_all(tlower:tupper,2:3)==j,1))==0
                banks_status_mtx(i,j)=1;
            end
        end
        
        % Restarts if the first quarter in the sample is not quarter two
        if i==1 && quarterstart~=2
            i=1;
        else
            i=i+1;
        end
        
        dd1=datenum(yearstart,7,1);
        dd2=datenum(yearstart,9,30);
        tlower = find(loans_all(:,1)>=dd1,1,'first');
        tupper = find(loans_all(:,1)<=dd2,1,'last');
        for j = 1:cols(banks_idx)
            if isempty(find(loans_all(tlower:tupper,2:3)==j,1))==0
                banks_status_mtx(i,j)=1;
            end
        end
        
        % Restarts if the first quarter in the sample is not quarter three
        if i==1 && quarterstart~=3
            i=1;
        else
            i=i+1;
        end
        
        dd1=datenum(yearstart,10,1);
        dd2=datenum(yearstart,12,31);
        tlower = find(loans_all(:,1)>=dd1,1,'first');
        tupper = find(loans_all(:,1)<=dd2,1,'last');
        for j = 1:cols(banks_idx)
            if isempty(find(loans_all(tlower:tupper,2:3)==j,1))==0
                banks_status_mtx(i,j)=1;
            end
        end
        
        yearstart = yearstart+1;
        i=i+1;
    end

%**********************************************************************
    
elseif frequency == 2 % (Monthly)
    
    % Construct matrix of start and end dates of each month in the sample
    datebeg = datenum(year(tradingdays(1)),month(tradingdays(1)),1);
    dateend = datenum(year(tradingdays(end)),month(tradingdays(end))+1,1)-1;
    mymonths = [];
    datenow = datebeg;
    while datenow < dateend
        mymonths = [mymonths; datenow, datenum(year(datenow),month(datenow)+1,1)-1];
        datenow = datenum(year(datenow),month(datenow)+1,1);
    end
    
    % Create monthly matrix of banks' status: active/inactive
    banks_status_mtx = zeros(rows(mymonths),cols(banks_idx));
    
    % Populate bank status matrix
    for i = 1:rows(mymonths)
        for q = 1:cols(banks_idx)                       
            if isempty(find((loans_all(:,1)>=mymonths(i,1) & loans_all(:,1)<=mymonths(i,2) & loans_all(:,2)==q) | ...
                            (loans_all(:,1)>=mymonths(i,1) & loans_all(:,1)<=mymonths(i,2) & loans_all(:,3)==q) ,1))==0 
                banks_status_mtx(i,q)=1;
            end
        end
    end
    
%********************************************************************** 
elseif frequency == 3 % (Weekly)
    
    % Construct matrix of start and end dates of each week in the sample
    datebeg = datenum(year(tradingdays(1)),month(tradingdays(1)),day(tradingdays(1)))-weekday(tradingdays(1))+1;
    dateend = datenum(year(tradingdays(end)),month(tradingdays(end)),day(tradingdays(end)))-weekday(tradingdays(end))+7;
    myweeks = [];
    datenow = datebeg;
    while datenow < dateend
        myweeks = [myweeks; datenow, datenow+6];
        datenow = datenow + 7;
    end
    
    % Create weekly matrix of banks' status: active/inactive
    banks_status_mtx = zeros(rows(myweeks),cols(banks_idx));    
    
    % Populate bank status matrix
    for i = 1:rows(myweeks)
        for q = 1:cols(banks_idx)                       
            if isempty(find((loans_all(:,1)>=myweeks(i,1) & loans_all(:,1)<=myweeks(i,2) & loans_all(:,2)==q) | ...
                            (loans_all(:,1)>=myweeks(i,1) & loans_all(:,1)<=myweeks(i,2) & loans_all(:,3)==q) ,1))==0 
                banks_status_mtx(i,q)=1;
            end
        end
    end
       
%**********************************************************************
elseif frequency == 4 %  (Daily)
    
    dtot=rows(tradingdays);
    
    % Create matrix of banks' status: active/inactive
    banks_status_mtx = zeros(dtot,cols(banks_idx));
    
    for i = 1:rows(tradingdays)
        for q = 1:cols(banks_idx)
            if isempty(find((loans_all(:,1)==tradingdays(i) & loans_all(:,2)==q) | (loans_all(:,1)==tradingdays(i) & loans_all(:,3)==q),1))==0 
                banks_status_mtx(i,q)=1;
            end
        end
    end
     
end

%% Deactivate IDs that have never traded 
% Do not eliminate IDs because other scripts assume the number of each 
% column in this matrix is equal to the ID of each bank.
banks_status_mtx(:,sum(banks_status_mtx)==0)=NaN;

%% Remove any empty networks
banks_status_mtx(sum(banks_status_mtx,2,'omitnan')==0,:) = [];

end
