%   The error function aggregates the number of inconsistencies 
%   between the observed network and a reference model
%
% ===================================
%   AUTHOR AND INSTITUTION INFO:
%   Written by Anthony Brassil and Gabriela Nodari
%   Reserve Bank of Australia
%   This version: January 2018
% ===================================
%   Available estimators:
%   1: CORRELATION
%   2: MAXIMUM LIKELIHOOD 
%   3: DENSITY-BASED - Brassil and Nodari (2018)
%   4: Craig and von Peter (2014)
% ======================================================================

function [error_tot]=errorfn(core,adj_mtx,vers)

N = size(adj_mtx,2);
Nc = sum(core==1);     % # of banks in the core
Np = N-Nc;             % # of banks in the periphery
c=find(core==1);       % position of banks in the adj matrix
Pcc=adj_mtx(c,c);      % Core block
adj_mtxtemp = adj_mtx;
adj_mtxtemp(c,:)=[];
adj_mtxtemp(:,c)=[];
Ppp = adj_mtxtemp;     % Periphery block
Pcp = adj_mtx(c,:);    % Core-Periphery block
Pcp(:,c) = [];
Ppc = adj_mtx(:,c);    % Periphery-Core block
Ppc(c,:) = [];
clear adj_mtxtemp

% ***********************************************************************
%                  CORRELATION METHOD (Section 4.1)
% ***********************************************************************
if vers == 1
    I_bar = (Nc*(Nc-1))/((Nc*(Nc-1))+(Np*(Np-1)));
    R_c_1 = sum(sum(Pcc));
    R_p_1 = sum(sum(Ppp));
    R_bar = (R_c_1+R_p_1)/((Nc*(Nc-1))+(Np*(Np-1)));
    R_c_0 = (Nc*(Nc-1))-R_c_1;
    R_p_0 = (Np*(Np-1))-R_p_1;
    numerator = (R_c_1*(1-R_bar)*(1-I_bar))+(R_c_0*(-R_bar)*(1-I_bar))+(R_p_1*(1-R_bar)*(-I_bar))+(R_p_0*(-R_bar)*(-I_bar));
    denom12 = ((R_c_1+R_p_1)*((1-R_bar)^2))+((R_c_0+R_p_0)*(R_bar^2));
    denom22 = ((R_c_1+R_c_0)*((1-I_bar)^2))+((R_p_1+R_p_0)*(I_bar^2));
    error_tot = -numerator/sqrt(denom12*denom22); % Maximise the correlation by minimising the negative

% ***********************************************************************
%                  MAXIMUM LIKELIHOOD (Section 4.2)
% ***********************************************************************

elseif vers == 2
    s_cc = Nc^2 - Nc;      % # of possible core-core links
    s_cp = Nc*Np;          % # of possible core-periphery links
    s_pc = s_cp;           % # of possible periphery-core links
    s_pp = Np^2 - Np;      % # of possible periphery-periphery links

% PS. We constrain s_xx to be different from zero in IntOptGlob.m by
% disregarding all cores of size 1 or N-1. However, the number of existing
% links (lambda_xx) could still be zero.

    lambda_cc = sum(Pcc(:));  % # of existing core-core links 
    lambda_cp = sum(Pcp(:));  % # of existing core-periphery links 
    lambda_pc = sum(Ppc(:));  % # of existing periphery-core links 
    lambda_pp = sum(Ppp(:));  % # of existing periphery-periphery links 

%===========================================================
% Adjustments required to prevent undefined log likelihood at the limits
% (see Footnote 32 in Appendix A).

    if lambda_cc==0
        Llambda_cc=1;
    else
        Llambda_cc=lambda_cc;
    end

    if lambda_cp==0
        Llambda_cp=1;
    else
        Llambda_cp=lambda_cp;
    end

    if lambda_pc==0
        Llambda_pc=1;
    else
        Llambda_pc=lambda_pc;
    end

    if lambda_pp==0
        Llambda_pp=1;
    else
        Llambda_pp=lambda_pp;
    end

    if s_cc==lambda_cc
        diff_s_lambda_cc=1;
    else
        diff_s_lambda_cc=s_cc-lambda_cc;
    end

    if s_cp==lambda_cp
       diff_s_lambda_cp=1;
    else
       diff_s_lambda_cp=s_cp-lambda_cp;
    end

    if s_pc==lambda_pc
       diff_s_lambda_pc=1;
    else
       diff_s_lambda_pc=s_pc-lambda_pc;
    end

    if s_pp==lambda_pp
       diff_s_lambda_pp=1;
    else
       diff_s_lambda_pp=s_pp-lambda_pp;
    end

%===========================================================
    
   logL = lambda_cc*log(Llambda_cc) + (s_cc-lambda_cc)*log(diff_s_lambda_cc)-(s_cc)*log(s_cc)...
          + lambda_pc*log(Llambda_pc) + (s_pc-lambda_pc)*log(diff_s_lambda_pc)-(s_pc)*log(s_pc)... 
          + lambda_cp*log(Llambda_cp) + (s_cp-lambda_cp)*log(diff_s_lambda_cp)-(s_cp)*log(s_cp)...       
          + lambda_pp*log(Llambda_pp) + (s_pp-lambda_pp)*log(diff_s_lambda_pp)-(s_pp)*log(s_pp);
      
    error_tot = -logL; % Maximise the log likelihood by minimising the negative

% ***********************************************************************
%     Density-Based estimator (Brassil and Nodari 2018) (Section 5)
% ***********************************************************************

elseif vers == 3
    
    if Np > 0
        e_pp = sum(Ppp(:))/(Np^2 - Np);
    else
        e_pp = 0;
    end
    
    if Nc > 0
        e_cc = 1 -(sum(Pcc(:))/(Nc^2 - Nc));
    else
        e_cc = 0;
    end
    
    if Np > 0 && Nc > 0
        e_cp = sum(sum(Pcp,2)==0)/Nc;
        e_pc = sum(sum(Ppc,1)==0)/Nc;
    else
        e_cp = 0;
        e_pc = 0;
    end
    
    error_tot = (e_cc + e_pp + e_cp + e_pc)/4; % Maximum error is one.

% ***********************************************************************
%               Craig and von Peter (2014) (Section 4.3)
% ***********************************************************************
elseif vers == 4 
    
    e_cc = (Nc^2 - Nc) - sum(Pcc(:));
    e_pp = sum(Ppp(:));
    e_cp = Np*sum(sum(Pcp,2)==0); 
    e_pc = Np*sum(sum(Ppc,1)==0);
    M = (size(adj_mtx,1))^2 - size(adj_mtx,1); % Constant used by CvP for normalising the error function

    error_tot = (e_cc + e_pp + e_cp + e_pc)/M;  
    
end