%% coreEst
%
% This function estimates the core and periphery of a series of networks.
%
% =========================================================
% AUTHOR AND INSTITUTION INFO:
% Written by Anthony Brassil and Gabriela Nodari 
% Reserve Bank of Australia
% This version: January 2018
% ==========================================================

function [CPvecGlob,CPvecAllGlob,coreSize_min,coreSize_max]=coreEst(banks_status_mtx,adj_ALL,Vers,Iter)

% Let N be the number of nodes that appear that appear in at least one of
% the networks. Let T be the total number of networks.

% banks_status_mtx is a TxN matrix of 1,0,NaN. Cell (t,i) takes a value of 
% 1 if Node i is active in network t, 0 if it is not active in network t 
% but is active in at least one network, NaN if it is never active.
% adj_ALL is a 1xT array of cells. Cell (1,t) contains the adjacency matrix
% of nodes in network t (recall that these estimators are designed
% for unweighted/directed networks, so the adjacency matrix contains zeros
% and ones). This includes inactive nodes (nodes with no edges).
% Vers is a scalar determining which estimator will be used (see errorfn.m
% for more details).
% Iter is a scalar defining the number of random starting points the
% optimisation algorithm will use (see IntOptGlob.m for more details).

% Create Cell array to store identified core(s) for each period
CPvecGlob = cell(1,size(adj_ALL,2));
CPvecAllGlob = cell(1,size(adj_ALL,2));

for t = 1:size(adj_ALL,2)
    nodes = banks_status_mtx(t,:);
    nodes(nodes~=1)=0; % IntOptGlob does not require information on whether the node is ever active
    
    % Estimate core/periphery structure
    [CPvec,~,~,~,CPvecAll] = IntOptGlob(adj_ALL{1,t},nodes,Vers,Iter);
    str = ['Network: ',num2str(t),'/',num2str(size(adj_ALL,2))];
    disp(str)
    
    % Store core(s) as rows
    CPvecGlob{1,t} = CPvec'; % If multiple cores found (with same minimum error), this stores a randomly chosen core
    CPvecAllGlob{1,t} = CPvecAll'; % Stores all identified core/periphery structures
end

coreSize_min = zeros(size(adj_ALL,2),1);
coreSize_max = zeros(size(adj_ALL,2),1);

for t=1:size(CPvecGlob,2)
    tmp = sum(CPvecAllGlob{1,t},2);
    coreSize_min(t,1) = min(tmp);
    coreSize_max(t,1) = max(tmp);
end

end