/**************************************************************
  Program created by James Bishop during RBA inposting in 2016. It uses
  the WPI micro data to estimate the effect of award wages on wages and hours 
  worked.  
  
  Estimates for the probability of job loss are derived in flat_dollar_transitions.do
**************************************************************/

clear all

set more off

cd "\\sasnasprd\SASData\W386\RBA\dta_files\master"

use master, clear

local awd_measure j_awardm2
* j_awardm1

keep if `awd_measure'==1 | j_eba_ddd==1 

drop if surveyq>=tq(2016q1)

/************************************************************************************
 Sample restrictions


 - Restrictions for 2006 to 2010 sample
   
   Because state and federal IRC policy diverged over this period, it is important to distinguish
   between whether the firm was covered by the federal or state system.  While this is difficult to 
   do, we can identify which firms are 'definitely' in the federal system. For the remaining firms, 
   it is unclear as to whether they are state or federal system, so these jobs are excluded from 
   the regression analysis. 
   
 - Exclude juniors, apprentices and trainees
 
   Paid a fixed proportion of a relevant adult wage rate (and hence will grow at the same rate as that
   adult rate). It is not easy to discern what the relevant adult rate is for these employees. And in 
   most cases, it will be the lowest classification level, which means that there will be little variation
   in minimum wages within juniors, apprentice and trainee categories (they will all receive close to the
   maximum percentage increase). 

*************************************************************************************/

* dummy=1 if definitely national system employer from 2006 to 2009, =0 otherwise
gen     national = 0

* Jobs in VIC, NT, ACT 
replace national = 1 if f_stateest==2 | f_stateest==7 | f_stateest==8

* Constitutional Corporations: 
* (a) Private corporations are unambiguously within the national system
replace national = 1 if f_mutolo>=1  & f_mutolo<=5

* (b) Unincorporated entities (sole traders, partnerships) are either federal or state. Assume 
*     federal if and only if the job ever had its pay set according to Australian Pay and Classification Scales (APCS)
gen                   ps    = 1 if m_pmekflg=="PS"
bysort f_unitid: egen ps_max=max(ps)
replace national = 1 if f_mutolo>=6 & f_mutolo<=8 & ps_max==1

* (c) Some legal structures are ambiguous in terms of their 'constitutional corporation' status e.g. trusts, 
*     charities, sporting clubs, local government. Again, use APCS to assign to federal system
replace national = 1 if f_mutolo>=9  & f_mutolo<=20 & ps_max==1
replace national = 1 if f_mutolo>=34 & f_mutolo<=35 & ps_max==1

* (d) All Commonwealth and Commonwealth authorities
replace national = 1 if f_mutolo>=21 & f_mutolo<=26

drop if (surveyq>=tq(2006q2) & surveyq<=tq(2010q1)) &  national!=1


* Drop apprentices, trainees and juniors

drop if j_j_apprentice==1  | j_j_trainee==1 | j_j_junior==1

sort  j_id surveyq
xtset j_id surveyq

/*************************************************************************************
    Wage measures

    Measure 1: raw provider data i.e. no adjustments
	
	[m_hbsep]      incorporates casual loading
	[m_hbsep_excs] excludes casual loading 

		
	Measure 2: all 'splice' adjustments, including:
	   - reporting error
	   - changes in qualifications/training/age/increment (inc. trainees & apprentices)
	
   [w_wage_spl1_dln]
       
*************************************************************************************/

drop w_wage_spl2_dln

rename m_hbsep      wage1
rename m_hbsep_excs wage2

forvalues c = 1(1)2 {

replace wage`c'     = . if wage`c'==0
gen     wage`c'_ln  = ln(wage`c')
}
*

/************************************************************************************
 Calculate award wage adjustment for each decision based on the hourly wage
 
 SNR adjustments were to the weekly award wage. 
 
 If job is full-time, base the SNR adjustment the job's weekly standard hours (i.e. multiply
 the hourly base wage by standard weekly hours to get the weekly award rate of pay).
 
 If job is part-time or casual (or if full-time indicator not available e.g. post-2003q2), assume 
 the full-time hours in the award are 38 hours per week.
 
 According to the Award Review Taskforce (Taskforce) Final Report on Rationalisation of Wage and 
 Classification Structures (2006, p. 80), ...299 pre-reform awards have been identified as having 
 classifications that are required to work more than 38 hours per week. http://pandora.nla.gov.au/tep/84669
 
 The maximum percentage increase is the increase that would be awarded to an employee on the FMW. For 
 employees whose actual wage is below the FMW, assume it is equal to the FMW for the purposes of calculating
 the award wage increase.
 
*************************************************************************************/
sort  j_id surveyq

replace  m_stdhrsd=j_wkstdhrs if m_stdhrsd==0

* If standrad hours are in a 'sensible range' for FT hours, use standard hours. Otherwise use 38 hours.

forvalues c = 1(1)2 { 
gen      week_awd`c' =    wage`c' * m_stdhrsd if    m_stdhrsd>=35 & m_stdhrsd<=44
replace  week_awd`c' =    wage`c' * 38        if    m_stdhrsd< 35 | m_stdhrsd> 44

* From 2006, the adjustments were to the hourly (rather than weekly) rate. Use conversion factor of 38 for all
replace  week_awd`c' =    wage`c' * 38        if    surveyq>=tq(2006q4)
}
*

* Federal Minimum Wage
gen     fmw = . 
replace fmw = 359.40 if surveyq<=tq(1998q2)
replace fmw = 373.40 if surveyq>=tq(1998q3) & surveyq<=tq(1999q2)
replace fmw = 385.40 if surveyq>=tq(1999q3) & surveyq<=tq(2000q2)
replace fmw = 400.40 if surveyq>=tq(2000q3) & surveyq<=tq(2001q2)
replace fmw = 413.40 if surveyq>=tq(2001q3) & surveyq<=tq(2002q2)
replace fmw = 431.40 if surveyq>=tq(2002q3) & surveyq<=tq(2003q2)
replace fmw = 448.40 if surveyq>=tq(2003q3) & surveyq<=tq(2004q2)
replace fmw = 467.40 if surveyq>=tq(2004q3) & surveyq<=tq(2005q2)

replace fmw = 484.40 if surveyq>=tq(2005q3) & surveyq<=tq(2006q4)
replace fmw = 511.80 if surveyq>=tq(2007q1) & surveyq<=tq(2007q3)
replace fmw = 522.12 if surveyq>=tq(2007q4) & surveyq<=tq(2008q3)
replace fmw = 543.78 if surveyq>=tq(2008q4) & surveyq<=tq(2010q2)

replace fmw = 569.90 if surveyq>=tq(2010q3) & surveyq<=tq(2011q2)
replace fmw = 589.30 if surveyq>=tq(2011q3) & surveyq<=tq(2012q2)
replace fmw = 606.40 if surveyq>=tq(2012q3) & surveyq<=tq(2013q2)
replace fmw = 622.20 if surveyq>=tq(2013q3) & surveyq<=tq(2014q2)
replace fmw = 640.90 if surveyq>=tq(2014q3) & surveyq<=tq(2015q2)

save transitions, replace


* assign the SNR increase to the quarter before it takes effect

forvalues c = 1(1)2 {

gen     snr`c' = .

replace snr`c' = 14 if week_awd`c' <  550                     & surveyq==tq(1998q2) & week_awd`c'!=.
replace snr`c' = 12 if week_awd`c' >= 550 & week_awd`c' < 700 & surveyq==tq(1998q2) & week_awd`c'!=. 
replace snr`c' = 10 if week_awd`c' >= 700                     & surveyq==tq(1998q2) & week_awd`c'!=. 

replace snr`c' = 12 if week_awd`c' <  510                     & surveyq==tq(1999q2) & week_awd`c'!=.
replace snr`c' = 10 if week_awd`c' >= 510                     & surveyq==tq(1999q2) & week_awd`c'!=.

replace snr`c' = 15 if                                          surveyq==tq(2000q2) & week_awd`c'!=. 

replace snr`c' = 13 if week_awd`c' <  490                     & surveyq==tq(2001q2) & week_awd`c'!=. 
replace snr`c' = 15 if week_awd`c' >= 490 & week_awd`c' < 590 & surveyq==tq(2001q2) & week_awd`c'!=. 
replace snr`c' = 17 if week_awd`c' >= 590                     & surveyq==tq(2001q2) & week_awd`c'!=. 

replace snr`c' = 18 if                                          surveyq==tq(2002q2) & week_awd`c'!=. 

replace snr`c' = 17 if week_awd`c' <  731.80                  & surveyq==tq(2003q2) & week_awd`c'!=. 
replace snr`c' = 15 if week_awd`c' >= 731.80                  & surveyq==tq(2003q2) & week_awd`c'!=. 

replace snr`c' = 19 if                                          surveyq==tq(2004q2) & week_awd`c'!=.

replace snr`c' = 17 if                                          surveyq==tq(2005q2) & week_awd`c'!=. 

replace snr`c' = 27.36 if week_awd`c' <  700                  & surveyq==tq(2006q4) & week_awd`c'!=.
replace snr`c' = 22.04 if week_awd`c' >= 700                  & surveyq==tq(2006q4) & week_awd`c'!=. 

replace snr`c' = 10.26 if week_awd`c' <  700                  & surveyq==tq(2007q3) & week_awd`c'!=. 
replace snr`c' =  5.30 if week_awd`c' >= 700                  & surveyq==tq(2007q3) & week_awd`c'!=. 

replace snr`c' = 21.66 if                                       surveyq==tq(2008q3) & week_awd`c'!=. 

* The SNR for 2009 is the amount that we assume for the placebo test. Assume the same as the 2008 decision
replace snr`c' = 21.66 if                                       surveyq==tq(2009q3) & week_awd`c'!=. 

replace snr`c' = 26.10 if                                       surveyq==tq(2010q2) & week_awd`c'!=. 


/**************************************************************
 Award adjustment
  
 If week_awd is less than FMW, assume it is equal to FMW for purposes of calculating the  
 percentage adjustment (since that was the largest possible percentage adjustment in each year).
**************************************************************/

* Predicted % increase in base wage under SNR

gen     week_awd_adj`c' = week_awd`c'
replace week_awd_adj`c' = fmw       if week_awd`c' < fmw & (week_awd`c'!=. & fmw!=.) 

sort    j_id surveyq

* Assigned to the survey immediately prior to the adjustment (i.e. May between 1998 and 2008)
gen     snr_adj_pre`c'  = ln(week_awd_adj`c'+snr`c') - ln(week_awd_adj`c')
replace snr_adj_pre`c'  = ln(week_awd_adj`c'*1.034)  - ln(week_awd_adj`c') if surveyq==tq(2011q2) & week_awd`c'!=.
replace snr_adj_pre`c'  = ln(week_awd_adj`c'*1.029)  - ln(week_awd_adj`c') if surveyq==tq(2012q2) & week_awd`c'!=.
replace snr_adj_pre`c'  = ln(week_awd_adj`c'*1.026)  - ln(week_awd_adj`c') if surveyq==tq(2013q2) & week_awd`c'!=.
replace snr_adj_pre`c'  = ln(week_awd_adj`c'*1.030)  - ln(week_awd_adj`c') if surveyq==tq(2014q2) & week_awd`c'!=.
replace snr_adj_pre`c'  = ln(week_awd_adj`c'*1.025)  - ln(week_awd_adj`c') if surveyq==tq(2015q2) & week_awd`c'!=.

* Assigned to the survey 6 months after the adjustment
gen     snr_adj_post`c' = L2.snr_adj_pre`c'

}
sort  j_id surveyq


/**************************************************************
* Split sample into different pre- and post-experiments
**************************************************************/

gen     treatment_no = .
* Workplace Relations Act:
replace treatment_no = 1  if surveyq==tq(1998q2) | surveyq==tq(1998q4)
replace treatment_no = 2  if surveyq==tq(1999q2) | surveyq==tq(1999q4)
replace treatment_no = 3  if surveyq==tq(2000q2) | surveyq==tq(2000q4)
replace treatment_no = 4  if surveyq==tq(2001q2) | surveyq==tq(2001q4)
replace treatment_no = 5  if surveyq==tq(2002q2) | surveyq==tq(2002q4)
replace treatment_no = 6  if surveyq==tq(2003q2) | surveyq==tq(2003q4)
replace treatment_no = 7  if surveyq==tq(2004q2) | surveyq==tq(2004q4)
replace treatment_no = 8  if surveyq==tq(2005q2) | surveyq==tq(2005q4)
* WorkChoices:
replace treatment_no = 9  if surveyq==tq(2006q4) | surveyq==tq(2007q2)
replace treatment_no = 10 if surveyq==tq(2007q3) | surveyq==tq(2008q1)
replace treatment_no = 11 if surveyq==tq(2008q3) | surveyq==tq(2009q1)
* Minimum wage Freeze:
replace treatment_no = 12 if surveyq==tq(2009q3) | surveyq==tq(2010q1)
* Modernisation, flat dollar increase:
replace treatment_no = 13 if surveyq==tq(2010q2) | surveyq==tq(2010q4)
* Modernisation, flat percentage increase:
replace treatment_no = 14 if surveyq==tq(2011q2) | surveyq==tq(2011q4)
replace treatment_no = 15 if surveyq==tq(2012q2) | surveyq==tq(2012q4)
replace treatment_no = 16 if surveyq==tq(2013q2) | surveyq==tq(2013q4)
replace treatment_no = 17 if surveyq==tq(2014q2) | surveyq==tq(2014q4)
replace treatment_no = 18 if surveyq==tq(2015q2) | surveyq==tq(2015q4)


* Treatment period dummy
gen     d = .
replace d = 0 if j_quarter==2 & ((treatment_no>=1  & treatment_no<=8) | (treatment_no>=13 & treatment_no<=18))
replace d = 0 if j_quarter==4 &   treatment_no==9 
replace d = 0 if j_quarter==3 &  (treatment_no>=10 & treatment_no<=12) 

replace d = 1 if j_quarter==4 & ((treatment_no>=1  & treatment_no<=8) | (treatment_no>=13 & treatment_no<=18))
replace d = 1 if j_quarter==2 &   treatment_no==9 
replace d = 1 if j_quarter==1 &  (treatment_no>=10 & treatment_no<=12) 


* Define treatment group (awd=1) as individuals paid award wage AT START OF TRANSITION
gen     awd = .
replace awd = 0        if j_eba_ddd==1     & treatment_no!=. & d==0 
replace awd = 1        if `awd_measure'==1 & treatment_no!=. & d==0 
replace awd = L2.awd if                      treatment_no!=. & d==1 


* quarter dummies
tab surveyq if treatment_no!=., gen(qtr_) 


/**************************************************************
* Define key variables (levels specification)
**************************************************************/

/* Log wage 
   
   For pre-periods, use wage1_ln
   
   For post-periods, grow forward wage1_ln by the cumulative log change in the spliced measure of wages
    between the pre- and post-period */

gen     ln_wage = wage1_ln                                           if treatment_no!=. & d==0 
replace ln_wage = L2.wage1_ln + L1.w_wage_spl1_dln + w_wage_spl1_dln if treatment_no!=. & d==1 

gen     dln_wage= w_wage_spl1_dln + L1.w_wage_spl1_dln  if d==1

* Unspliced measure

gen     dln_wage_unspliced= wage1_ln - L2.wage1_ln if d==1

* Hours Worked (conditional on employment)

gen    ln_hrs = ln(m_ordhrswk_jb) if d==0 | d==1

gen    dln_hrs= ln(m_ordhrswk_jb) - ln(L2.m_ordhrswk_jb) if d==1

gen    dhrs = m_ordhrswk_jb - L2.m_ordhrswk_jb if d==1

* Flag for receiving overtime payment in the quarter

gen     overtime = 0 if m_vpayflg!=""
replace overtime = 1 if m_vpayflg=="Y"

* Employee churn within jobs (=1 if job occupant changed over previous 6 months)
* Ensure that it is only for a balanced panel - the second measure (dif_emp2)

gen     dif_emp = 0 if  j_same_emp==1 & L1.j_same_emp==1
replace dif_emp = 1 if (j_same_emp==0 | L1.j_same_emp==0) & j_same_emp!=. & j_same_emp!=.

gen     dif_emp2 = dif_emp
replace dif_emp2 = . if F2.dif_emp==. & d==0
replace dif_emp2 = . if L2.dif_emp==. & d==1

* Change in grade/class/level (=1 if level changed over previous 6 months)
* Ensure that it is only for a balanced panel - the second measure (dif_emp2)

gen     dif_lev = 0 if  j_same_lev==1 & L1.j_same_lev==1
replace dif_lev = 1 if (j_same_lev==0 | L1.j_same_lev==0) & j_same_lev!=. & j_same_lev!=.

gen     dif_lev2 = dif_lev
replace dif_lev2 = . if F2.dif_lev==. & d==0
replace dif_lev2 = . if L2.dif_lev==. & d==1

/**************************************************************
  Drop public sector jobs
**************************************************************/

tab e_public awd if  j_quarter==2 & treatment_no!=.

keep if e_public==0

save working_new, replace

/**************************************************************
 Construct wage decile dummies

  The 'control decile' is zero in the pre- and post-treatment period. 
  
  'Treatment deciles' are zero in the pre-treatment period and 1 in the post-treatment period.
 
**************************************************************/

sort  j_id surveyq

gen   F2_ln_wage=F2.ln_wage
gen   F2_ln_hrs =F2.ln_hrs


* Calculate deciles of award wage distribution, by year 
forvalues s=10(10)90 {
bysort treatment_no: egen p`s' = pctile(week_awd_adj2) if F2_ln_wage!=. & F2_ln_hrs!=. & awd==1 & d==0 & treatment_no!=., p(`s')

* Also assign these deciles to the post survey
bysort treatment_no: egen p`s'a = max(p`s') if treatment_no!=.
drop p`s'
rename p`s'a  p`s'
}
*

* Calculate wage decile categorical variable (1-10) 
* Include EBA jobs, but using the deciles of the award wage distribution rather than EBA distribution 

* The inclusion of the 'F2_ln_wage!=.' condition below, restricts the analysis to a set of balanced panels

sort  j_id surveyq

gen dec=.

forvalues x = 1(1)18 {
gen     dec`x' = .
replace dec`x' = 1 if week_awd_adj2< p10                      & week_awd_adj2!=. & F2_ln_wage!=. & F2_ln_hrs!=. & d==0 & treatment_no==`x' 
replace dec`x' = 2 if week_awd_adj2>=p10 & week_awd_adj2< p20 & week_awd_adj2!=. & F2_ln_wage!=. & F2_ln_hrs!=. & d==0 & treatment_no==`x'
replace dec`x' = 3 if week_awd_adj2>=p20 & week_awd_adj2< p30 & week_awd_adj2!=. & F2_ln_wage!=. & F2_ln_hrs!=. & d==0 & treatment_no==`x'
replace dec`x' = 4 if week_awd_adj2>=p30 & week_awd_adj2< p40 & week_awd_adj2!=. & F2_ln_wage!=. & F2_ln_hrs!=. & d==0 & treatment_no==`x'
replace dec`x' = 5 if week_awd_adj2>=p40 & week_awd_adj2< p50 & week_awd_adj2!=. & F2_ln_wage!=. & F2_ln_hrs!=. & d==0 & treatment_no==`x'
replace dec`x' = 6 if week_awd_adj2>=p50 & week_awd_adj2< p60 & week_awd_adj2!=. & F2_ln_wage!=. & F2_ln_hrs!=. & d==0 & treatment_no==`x'
replace dec`x' = 7 if week_awd_adj2>=p60 & week_awd_adj2< p70 & week_awd_adj2!=. & F2_ln_wage!=. & F2_ln_hrs!=. & d==0 & treatment_no==`x'
replace dec`x' = 8 if week_awd_adj2>=p70 & week_awd_adj2< p80 & week_awd_adj2!=. & F2_ln_wage!=. & F2_ln_hrs!=. & d==0 & treatment_no==`x'
replace dec`x' = 9 if week_awd_adj2>=p80 & week_awd_adj2< p90 & week_awd_adj2!=. & F2_ln_wage!=. & F2_ln_hrs!=. & d==0 & treatment_no==`x'
replace dec`x' =10 if week_awd_adj2>=p90                      & week_awd_adj2!=. & F2_ln_wage!=. & F2_ln_hrs!=. & d==0 & treatment_no==`x'

replace dec`x' = L2.dec`x' if treatment_no==`x' & d==1

replace dec=dec`x' if treatment_no==`x'
}
*

* Number of jobs in each decile (to check if have enough EBA jobs in each decile)

tab dec if awd==1 & F2_ln_wage!=. & d==0 & treatment_no!=.
tab dec if awd==0 & F2_ln_wage!=. & d==0 & treatment_no!=.

sort  j_id surveyq


* define omitted categories (by dropping dec10)
* this will also establish the relevant omitted categories for the interactions below
tab dec, gen(dec_)
drop dec_10

/**************************************************************
  Interaction terms for DD and DDD models
**************************************************************/

* Treatment period dummy X wage decile 
forvalues c = 1(1)9 {
gen dXdec_`c'=d*dec_`c'
}
* Treatment_no X wage decile (allows for time-varying group-level heterogeneity)
tab treatment_no, gen(tr_)
forvalues p = 1(1)18  {
forvalues c = 1(1)9 {
gen tr_`p'Xdec_`c'=tr_`p'*dec_`c'
}
}
* awd X quarter  
forvalues c = 1(1)36 {
gen awdXqtr_`c'=awd*qtr_`c'
}
* awd X wage decile  
forvalues c = 1(1)9 {
gen awdXdec_`c'=awd*dec_`c'
}
* awd X treatment_no X wage decile 
forvalues p = 1(1)18  {
forvalues c = 1(1)9 {
gen awdXtr_`p'Xdec_`c'=tr_`p'*dec_`c'*awd
}
drop tr_`p'
}
* awd X treatment period dummy X wage decile (DDD coefficient in pooled model)
forvalues c = 1(1)9 {
gen awdXdXdec_`c'=awd*d*dec_`c'
}
*

/**************************************************************
 Variables in continuous treatment specification
 
 For placebo period (2015), construct treatment by assuming that the dollar adjustment
 to the FMW (42 cents p/hr or $16 per week) was uniformly applied to all awards
**************************************************************/

* Continuous treatment specification (DD)
gen      c_treat = snr_adj_post2 if d==1 
replace  c_treat = snr_adj_pre2  if d==0 
replace  c_treat = ln(week_awd_adj2+16) - ln(week_awd_adj2)  if surveyq==tq(2015q2)
replace  c_treat = L2.c_treat                                if surveyq==tq(2015q4)

gen dXc_treat = d * c_treat

* Treatment_no X c_treat (allows for time-varying group-level heterogeneity)
tab treatment_no, gen(tr_)
forvalues p = 1(1)18  {
gen c_treatXtr_`p'=tr_`p'*c_treat
drop tr_`p'
}
*
* Continuous treatment specification (DDD)
gen awdXc_treat = awd * c_treat
gen awdXdXc_treat = awd * d * c_treat


* Treatment_no X c_treat (allows for time-varying group-level heterogeneity in DDD)
tab treatment_no, gen(tr_)
forvalues p = 1(1)18  {
gen c_treatXawdXtr_`p'=tr_`p'*c_treat*awd
drop tr_`p'
}
*
save private, replace


/**************************************************************
 Descriptive statistics
 
 Calculate real wage in $2009q1 using All groups CPI, seasonally adjusted
***************************************************************/
clear

cd "\\sasnasprd\SASData\W386\RBA\data\other_data\CPI"

insheet using cpi.csv

gen surveyq = yq(year, quarter)
format surveyq %tq
drop year quarter
sort surveyq

* express cpi in decimal form and re-reference to 2009q1 

gen cpi_dec = cpi_sa / 92.5
drop cpi_sa

cd "\\sasnasprd\SASData\W386\RBA\dta_files\master"

save cpi, replace

use private, clear

merge m:1 surveyq using cpi

drop if _merge==2
drop _merge

* Real wage, deflated using CPI and assuming 38 hours per week

gen real_w = (week_awd_adj2 / 38) / cpi_dec

gen full = 1 if treatment_no>=1 & treatment_no<=11

sort j_id surveyq
bysort j_id : carryforward f_empl, gen(empl)

tab f_in1_17, gen(anz93_)
tab j_asco1 if j_asco1!=0, gen(occ_)

* Other (i.e. low-award) industries
gen anz93_other = anz93_1 + anz93_2 + anz93_4 + anz93_5 + anz93_6 + anz93_9 + anz93_10 + anz93_11 + anz93_13 + anz93_14

* Occupational skill groups (see ANZSIC93 documentation)
gen skill_1 = occ_1 + occ_2
gen skill_2 = occ_3
gen skill_3 = occ_4 + occ_5
gen skill_4 = occ_6 + occ_7
gen skill_5 = occ_8 + occ_9

tabstat c_treat dln_wage dln_hrs real_w j_casual overtime m_ordhrswk_jb empl anz93_* occ_* skill_* if treatment_no!=. & awd==1 & full==1 & dec!=. & d==0, by(dec) statistics(mean sd N) 
tabstat c_treat dln_wage dln_hrs real_w j_casual overtime m_ordhrswk_jb empl anz93_* occ_* skill_* if treatment_no!=. & awd==1 & full==1 & dec!=. & d==1, by(dec) statistics(mean sd N) 

drop anz93_* occ_* skill_* 

* To generate data for the scatter graphs:
bysort treatment_no: tabstat c_treat  if treatment_no!=. & awd==1 & full==1 & dec!=. & d==1, by(dec) statistics(mean sd N) 
bysort treatment_no: tabstat dln_wage if treatment_no!=. & awd==1 & full==1 & dec!=. & d==1, by(dec) statistics(mean sd N) 
bysort treatment_no: tabstat dln_hrs  if treatment_no!=.  & awd==1 & full==1 & dec!=. & d==1, by(dec) statistics(mean sd N) 

bysort treatment_no: tabstat c_treat  if treatment_no!=. & awd==0 & full==1 & dec!=. & d==1, by(dec) statistics(mean sd N) 
bysort treatment_no: tabstat dln_wage if treatment_no!=. & awd==0 & full==1 & dec!=. & d==1, by(dec) statistics(mean sd N) 
bysort treatment_no: tabstat dln_hrs  if treatment_no!=. & awd==0 & full==1 & dec!=. & d==1, by(dec) statistics(mean sd N) 


* Histogram of c_treat (demeaned for each year)

bysort surveyq: egen c_treat_mn = mean(c_treat) if treatment_no!=. & awd==1 & full==1 & dec!=. & d==1
gen c_treat_dmn = c_treat - c_treat_mn

twoway__histogram_gen c_treat_dmn if treatment_no!=. & awd==1 & full==1 & dec!=. & d==1, width(0.001) gen(density bin)

tabstat c_treat dln_wage week_awd_adj2 real_w j_casual overtime m_ordhrswk_jb empl if treatment_no!=. & awd==1 & full==1 & dec!=. & d==1, by(dec) statistics(sd)
tabstat c_treat dln_wage week_awd_adj2 real_w j_casual overtime m_ordhrswk_jb empl if treatment_no!=. & awd==1 & full==1 & dec!=. & d==1, by(dec) statistics(N)


/**************************************************************
 Regression analysis
***************************************************************/

cd "\\sasnasprd\SASData\W386\RBA\dta_files\master"

use private, clear

sort j_id surveyq
 
gen full = 1 if treatment_no>=1 & treatment_no<=11
gen plac = 1 if treatment_no==12
gen pmek = 1 if treatment_no>=5 & treatment_no<=11
 
local clust j_id
* dec
* Clustering on j_id leads to much more conservative SEs than clustering on deciles

set matsize 1000

cd "\\sasnasprd\SASData\W386\RBA\analysis\STATA_output"
 
* Also construct estimates for hours worked that exclude extreme values > or < 4SDs from the same mean growth rate 
 
sum dln_hrs if treatment_no>=1 & treatment_no<=12 & awd==1 & dec!=. & d==1, detail

gen dln_hrsz = (dln_hrs-r(mean))/r(sd) if treatment_no>=1 & treatment_no<=12 & dec!=. & d==1

gen     outlier = 0 if dln_hrsz!=.
replace outlier = 1 if abs(dln_hrsz)>4 & dln_hrsz!=.

bysort j_id treatment_no: egen outlier_max = max(outlier)

gen     ln_hrs_out = ln_hrs
replace ln_hrs_out = .      if outlier_max==1

gen     dln_hrs_out = dln_hrs
replace dln_hrs_out = .      if outlier_max==1

bysort treatment_no: tabstat dln_hrs_out  if treatment_no!=. & awd==1 & full==1 & dec!=. & d==1, by(dec) statistics(mean sd N) 
bysort treatment_no: tabstat dln_hrs_out  if treatment_no!=. & awd==0 & full==1 & dec!=. & d==1, by(dec) statistics(mean sd N) 


* Baseline DD/DDD models

sort j_id surveyq

foreach s in full plac pmek {

matrix A`s' = J(19,18,.)
matrix colnames A`s' = dd_w_a dd_w_e ddd_w dd_h_a dd_h_e ddd_h dd_o_a dd_o_e ddd_o dd_de_a dd_de_e ddd_de dd_dl_a dd_dl_e ddd_dl dd_ho_a dd_ho_e ddd_ho
matrix rownames A`s' = b_1 b_2 b_3 b_4 b_5 b_6 b_7 b_8 b_9 se_1 se_2 se_3 se_4 se_5 se_6 se_7 se_8 se_9 obs

matrix B`s' = J(3,18,.)
matrix colnames B`s' = dd_w_a dd_w_e ddd_w dd_h_a dd_h_e ddd_h dd_o_a dd_o_e ddd_o dd_de_a dd_de_e ddd_de dd_dl_a dd_dl_e ddd_dl dd_ho_a dd_ho_e ddd_ho
matrix rownames B`s' = b_ se_ obs

local i = 0

foreach x in ln_wage ln_hrs overtime dif_emp2 dif_lev2 ln_hrs_out {

* Percentile treatments

quietly reg `x' qtr_* dec_* dXdec_*                                      if treatment_no!=. & awd==1 & `s'==1 & dec!=., cluster(`clust') noomitted
forvalues z = 1(1)9 {
matrix A`s'[`z'  ,1+`i'] =  _b[dXdec_`z']*100
matrix A`s'[`z'+9,1+`i'] =  _se[dXdec_`z']*100
matrix A`s'[19,   1+`i'] = e(N)
}
quietly reg `x' qtr_* dec_* dXdec_*                                      if treatment_no!=. & awd==0 & `s'==1 & dec!=., cluster(`clust') noomitted
forvalues z = 1(1)9 {
matrix A`s'[`z'  ,2+`i'] =  _b[dXdec_`z']*100
matrix A`s'[`z'+9,2+`i'] =  _se[dXdec_`z']*100
matrix A`s'[19,   2+`i'] = e(N)
}
quietly reg `x' qtr_* dec_* dXdec_* awd awdXqtr_* awdXdec_* awdXdXdec_* if treatment_no!=. & `s'==1 & dec!=., cluster(`clust') noomitted
forvalues z = 1(1)9 {
matrix A`s'[`z'  ,3+`i'] =  _b[awdXdXdec_`z']*100
matrix A`s'[`z'+9,3+`i'] =  _se[awdXdXdec_`z']*100
matrix A`s'[19,   3+`i'] = e(N)
}
*

* Continuous treatment

quietly reg `x' qtr_* c_treat dXc_treat                                         if treatment_no!=. & awd==1 & `s'==1 & dec!=. , cluster(`clust') noomitted
matrix B`s'[1,1+`i'] =  _b[dXc_treat]
matrix B`s'[2,1+`i'] =  _se[dXc_treat]
matrix B`s'[3,1+`i'] = e(N)

quietly reg `x' qtr_* c_treat dXc_treat                                         if treatment_no!=. & awd==0 & `s'==1 & dec!=. , cluster(`clust') noomitted
matrix B`s'[1,2+`i'] =  _b[dXc_treat]
matrix B`s'[2,2+`i'] =  _se[dXc_treat]
matrix B`s'[3,2+`i'] = e(N)

quietly reg `x' qtr_* c_treat dXc_treat awd awdXqtr_* awdXc_treat awdXdXc_treat if treatment_no!=.          & `s'==1 & dec!=. , cluster(`clust') noomitted
matrix B`s'[1,3+`i'] =  _b[awdXdXc_treat]
matrix B`s'[2,3+`i'] =  _se[awdXdXc_treat]
matrix B`s'[3,3+`i'] = e(N)

local i = `i'+3
}
}
*

putexcel A1=matrix(Afull, names) using results.xlsx, sheet("decile_full") replace 
putexcel A1=matrix(Bfull, names) using results.xlsx, sheet("contin_full") modify 
putexcel A1=matrix(Aplac, names) using results.xlsx, sheet("decile_placebo") modify 
putexcel A1=matrix(Bplac, names) using results.xlsx, sheet("contin_placebo") modify 
putexcel A1=matrix(Apmek, names) using results.xlsx, sheet("decile_pmek") modify 
putexcel A1=matrix(Bpmek, names) using results.xlsx, sheet("contin_pmek") modify 


/************************************************************************************
 Estimates by year
*************************************************************************************/

foreach i in dd ddd {
matrix T_`i' = J(12,15,.)
matrix colnames T_`i' = w_b w_se w_N h_b h_se h_N o_b o_se o_N de_b de_se de_N ho_b ho_se ho_N
matrix rownames T_`i' = 1998 1999 2000 2001 2002 2003 2004 2005 2006 2007 2008 2009
}
*

local i = 0

foreach z in ln_wage ln_hrs overtime dif_emp2 ln_hrs_out {

* DD estimates by year (seperate regressions)
forvalues x = 1(1)12 {
quietly reg `z' d c_treat dXc_treat if treatment_no==`x' & awd==1 & dec!=., cluster(`clust') noomitted
matrix T_dd[`x',1+`i'] =  _b[dXc_treat]
matrix T_dd[`x',2+`i'] =  _se[dXc_treat]
matrix T_dd[`x',3+`i'] = e(N)
}

* DDD estimates by year (seperate regressions)
forvalues x = 1(1)12 {
quietly reg `z' d c_treat dXc_treat awd awd#d awdXc_treat awdXdXc_treat if treatment_no==`x' & dec!=., cluster(`clust') noomitted
matrix T_ddd[`x',1+`i'] =  _b[awdXdXc_treat]
matrix T_ddd[`x',2+`i'] =  _se[awdXdXc_treat]
matrix T_ddd[`x',3+`i'] = e(N)

}
local i = `i'+3
}
*

putexcel A1=matrix(T_dd , names) using results.xlsx, sheet("timevarying_dd")  modify 
putexcel A1=matrix(T_ddd, names) using results.xlsx, sheet("timevarying_ddd") modify 


* Hypothesis tests (ignore the placebo period). Test that elasticities are equal over time.

local clust j_id

* Treatment_no X c_treat X d 
tab treatment_no, gen(tr_)
forvalues p = 1(1)18  {
gen dXc_treatXtr_`p'=dXc_treat*tr_`p'
gen awdXc_treatXtr_`p'=awdXc_treat*tr_`p'
gen awdXdXc_treatXtr_`p' = awdXdXc_treat*tr_`p' 
drop tr_`p'
}
*

foreach z in ln_wage ln_hrs overtime dif_emp2 ln_hrs_out {
quietly reg `z' qtr_* c_treat c_treatXtr_* dXc_treat dXc_treatXtr_* awd awdXqtr_* awdXc_treat awdXc_treatXtr_* awdXdXc_treat awdXdXc_treatXtr_3-awdXdXc_treatXtr_11 if treatment_no!=.  & full==1 & dec!=. , cluster(`clust') noomitted
test (awdXdXc_treatXtr_3 awdXdXc_treatXtr_4 awdXdXc_treatXtr_5 awdXdXc_treatXtr_6 awdXdXc_treatXtr_7 awdXdXc_treatXtr_8 awdXdXc_treatXtr_9 awdXdXc_treatXtr_10 awdXdXc_treatXtr_11)
}
*

drop dXc_treatXtr_* awdXc_treatXtr_* awdXdXc_treatXtr_*


/************************************************************************************
 Robustness tests
 
 1- Allow for time-varying group-level heterogeneity in decile treatment specification
 
    [add tr_* and awdXtr_* to the percentile treatment specification]

	[add c_treatXtr_* and c_treatXawdXtr_* to the continuous treatment specification]

 2- Is continuous specification robust to controlling for decile dummies
 
 3- Excluding outliers (in terms of log changes)
 
 4- Introduce controls
*************************************************************************************/

* 1 - Allow for time-varying group-level heterogeneity 

foreach s in full plac pmek {

matrix A`s' = J(19,18,.)
matrix colnames A`s' = dd_w_a dd_w_e ddd_w dd_h_a dd_h_e ddd_h dd_o_a dd_o_e ddd_o dd_de_a dd_de_e ddd_de dd_dl_a dd_dl_e ddd_dl dd_ho_a dd_ho_e ddd_ho
matrix rownames A`s' = b_1 b_2 b_3 b_4 b_5 b_6 b_7 b_8 b_9 se_1 se_2 se_3 se_4 se_5 se_6 se_7 se_8 se_9 obs

matrix B`s' = J(3,18,.)
matrix colnames B`s' = dd_w_a dd_w_e ddd_w dd_h_a dd_h_e ddd_h dd_o_a dd_o_e ddd_o dd_de_a dd_de_e ddd_de dd_dl_a dd_dl_e ddd_dl dd_ho_a dd_ho_e ddd_ho
matrix rownames B`s' = b_ se_ obs

local i = 0

foreach x in ln_wage ln_hrs overtime dif_emp2 dif_lev2 ln_hrs_out {

* Percentile treatments

quietly reg `x' qtr_* dec_* tr_* dXdec_*                                      if treatment_no!=. & awd==1 & `s'==1 & dec!=., cluster(`clust') noomitted
forvalues z = 1(1)9 {
matrix A`s'[`z'  ,1+`i'] =  _b[dXdec_`z']*100
matrix A`s'[`z'+9,1+`i'] =  _se[dXdec_`z']*100
matrix A`s'[19,   1+`i'] = e(N)
}
quietly reg `x' qtr_* dec_* tr_* dXdec_*                                      if treatment_no!=. & awd==0 & `s'==1 & dec!=., cluster(`clust') noomitted
forvalues z = 1(1)9 {
matrix A`s'[`z'  ,2+`i'] =  _b[dXdec_`z']*100
matrix A`s'[`z'+9,2+`i'] =  _se[dXdec_`z']*100
matrix A`s'[19,   2+`i'] = e(N)
}
quietly reg `x' qtr_* dec_* tr_* dXdec_* awd awdXqtr_* awdXdec_* awdXtr_* awdXdXdec_* if treatment_no!=. & `s'==1 & dec!=., cluster(`clust') noomitted
forvalues z = 1(1)9 {
matrix A`s'[`z'  ,3+`i'] =  _b[awdXdXdec_`z']*100
matrix A`s'[`z'+9,3+`i'] =  _se[awdXdXdec_`z']*100
matrix A`s'[19,   3+`i'] = e(N)
}
*

* Continuous treatment

quietly reg `x' qtr_* c_treat c_treatXtr_* dXc_treat                                         if treatment_no!=. & awd==1 & `s'==1 & dec!=. , cluster(`clust') noomitted
matrix B`s'[1,1+`i'] =  _b[dXc_treat]
matrix B`s'[2,1+`i'] =  _se[dXc_treat]
matrix B`s'[3,1+`i'] = e(N)

quietly reg `x' qtr_* c_treat c_treatXtr_* dXc_treat                                         if treatment_no!=. & awd==0 & `s'==1 & dec!=. , cluster(`clust') noomitted
matrix B`s'[1,2+`i'] =  _b[dXc_treat]
matrix B`s'[2,2+`i'] =  _se[dXc_treat]
matrix B`s'[3,2+`i'] = e(N)

quietly reg `x' qtr_* c_treat c_treatXtr_* dXc_treat c_treatXawdXtr_* awd awdXqtr_* awdXc_treat awdXdXc_treat if treatment_no!=. & `s'==1 & dec!=. , cluster(`clust') noomitted
matrix B`s'[1,3+`i'] =  _b[awdXdXc_treat]
matrix B`s'[2,3+`i'] =  _se[awdXdXc_treat]
matrix B`s'[3,3+`i'] = e(N)

local i = `i'+3
}
}
*
putexcel A1=matrix(Afull, names) using results.xlsx, sheet("decile_full_tvuh") modify 
putexcel A1=matrix(Bfull, names) using results.xlsx, sheet("contin_full_tvuh") modify 
putexcel A1=matrix(Aplac, names) using results.xlsx, sheet("decile_placebo_tvuh") modify 
putexcel A1=matrix(Bplac, names) using results.xlsx, sheet("contin_placebo_tvuh") modify 
putexcel A1=matrix(Apmek, names) using results.xlsx, sheet("decile_pmek_tvuh") modify 
putexcel A1=matrix(Bpmek, names) using results.xlsx, sheet("contin_pmek_tvuh") modify 


/******************************************************************************************
 Include additional controls (firm ID)
 
 Firm FE controls for lots of possible shocks e.g. shocks to low minimum wage states, low minimum wage industries or small firms. 


 There are 4 equivalent approaches to introducing controls (using asco3_b as an example)

Approach 1
reg dln_wage qtr_* i.asco3_b c_treat     if treatment_no!=. & treatment_no<=11 & awd==1 & dec!=., cluster(j_id) noomitted

Approach 2
areg dln_wage qtr_* c_treat     if treatment_no!=. & treatment_no<=11 & awd==1 & dec!=., cluster(j_id) noomitted absorb(asco3_b)

Approach 3
reg ln_wage  qtr_* i.asco3_b d#asco3_b c_treat c_treatXtr_* dXc_treat if treatment_no!=. & awd==1 & treatment_no<=11 & dec!=., cluster(j_id) noomitted

Approach 4
egen d_asco3_b = group(d asco3_b)
areg ln_wage  qtr_* i.asco3_b  c_treat c_treatXtr_* dXc_treat if treatment_no!=. & awd==1 & treatment_no<=11 & dec!=., cluster(j_id) noomitted absorb(d_asco3_b)

******************************************************************************************/

encode f_unitid, gen(firm)

egen firm_by_d = group(d firm)

local clust j_id

foreach s in full pmek plac {

matrix A`s' = J(19,3,.)
matrix colnames A`s' = dd_w_a  dd_h_a dd_ho_a
matrix rownames A`s' = b_1 b_2 b_3 b_4 b_5 b_6 b_7 b_8 b_9 se_1 se_2 se_3 se_4 se_5 se_6 se_7 se_8 se_9 obs

matrix B`s' = J(3,3,.)
matrix colnames B`s' = dd_w_a dd_h_a dd_ho_a
matrix rownames B`s' = b_ se_ obs

local i = 0

foreach x in ln_wage ln_hrs ln_hrs_out {

* Percentile treatments (log-change specification)

quietly areg `x' qtr_* dec_* dXdec_*   if treatment_no!=. & `s'==1  & awd==1 & dec!=., cluster(`clust') noomitted absorb(firm_by_d)
forvalues z = 1(1)9 {
matrix A`s'[`z'  ,1+`i'] =  _b[dXdec_`z']*100
matrix A`s'[`z'+9,1+`i'] =  _se[dXdec_`z']*100
matrix A`s'[19,   1+`i'] = e(N)
}

* Continuous treatment (log-change specification)

quietly areg `x' qtr_* c_treat dXc_treat if treatment_no!=. & `s'==1 & awd==1 & dec!=., cluster(`clust') noomitted absorb(firm_by_d)
matrix B`s'[1,1+`i'] =  _b[dXc_treat]
matrix B`s'[2,1+`i'] =  _se[dXc_treat]
matrix B`s'[3,1+`i'] = e(N)

local i = `i'+1
}
}
*

putexcel A1=matrix(Afull, names) using results.xlsx, sheet("decile_full_controls") modify 
putexcel A1=matrix(Bfull, names) using results.xlsx, sheet("contin_full_controls") modify 
putexcel A1=matrix(Apmek, names) using results.xlsx, sheet("decile_pmek_controls") modify 
putexcel A1=matrix(Bpmek, names) using results.xlsx, sheet("contin_pmek_controls") modify 
putexcel A1=matrix(Aplac, names) using results.xlsx, sheet("decile_placebo_controls") modify 
putexcel A1=matrix(Bplac, names) using results.xlsx, sheet("contin_placebo_controls") modify 


/******************************************************************************************
 Robustness checks and additional results
******************************************************************************************/

* Is continuous specification robust to including controls for wage decile?

reg ln_wage qtr_* c_treat dec_* dXc_treat  if treatment_no!=. & awd==1 & full==1 & dec!=. , cluster(`clust') noomitted

* Exclude Queensland (due to timing issues)

foreach x in ln_wage ln_hrs {
reg `x' qtr_* c_treat      dXc_treat  if treatment_no!=. & awd==1 & full==1 & dec!=. & f_stateest!=3, cluster(j_id) noomitted
reg `x' qtr_* c_treat      dXc_treat  if treatment_no!=. & awd==0 & full==1 & dec!=. & f_stateest!=3, cluster(j_id) noomitted
reg `x' qtr_* c_treat dXc_treat awd awdXqtr_* awdXc_treat awdXdXc_treat if treatment_no!=.  & full==1 & dec!=. & f_stateest!=3, cluster(j_id) noomitted
}
*

* Unspliced wage measure. Also include dummy for whether grade/level change in previous 6 months, interacted with d
sort j_id surveyq

gen dif_lev2Xd     = dif_lev2 * d 
gen dif_lev2XdXawd = dif_lev2 * d * awd

reg wage1_ln qtr_* c_treat dif_lev2Xd     dXc_treat  if treatment_no!=. & awd==1 & full==1 & dec!=., cluster(j_id) noomitted
reg wage1_ln qtr_* c_treat dif_lev2Xd     dXc_treat  if treatment_no!=. & awd==0 & full==1 & dec!=., cluster(j_id) noomitted
reg wage1_ln qtr_* c_treat dif_lev2Xd dif_lev2XdXawd dXc_treat awd awdXqtr_* awdXc_treat awdXdXc_treat if treatment_no!=.  & full==1 & dec!=., cluster(j_id) noomitted


* end of do file

