%% plot_figure_C1

% =========================================================
% AUTHORS AND INSTITUTION INFO:
% Code written by Anthony Brassil and Gabriela Nodari
% Reserve Bank of Australia
% This version: January 2018
% ==========================================================

% This code runs numerical simulations to determine the accuracy of the
% density-based estimator relative to the Cucuringu et al (2016) estimator.
% Simulations are run for multiple core sizes.

%% Initialisation
clear all
clc
close all

%% Parameters
Smpl = 10000; % Number of random draws for each parameterisation
N = 40; % Number of nodes (calibrated to number of active banks in the data)
Iter = 20; % Number of random starting points for greedy (steepest descent) algorithm
DT = 0.25; % Density of the entire network (calibrate to average density of active banks in the data)
CN = [0; 0.05; 0.1; 0.15; 0.2; 0.25; 0.3; 0.35; 0.4; 0.45]; % Proportion of N that is in the core
vnames = cellstr(['DB ';'Cuc']); % Abbreviations for the estimators
ci = 0.01; % Set to 0.01 to construct the 99th percentile

NumEdg = round(N*(N-1)*DT); % Edges in this network
filename1 = ['Simulation_tmp_Cuc_',num2str(Smpl),'.mat'];
filename2 = ['Simulation_Cuc_',num2str(Smpl),'.mat'];

%% Construct simulated networks and initialise arrays and loops

if exist(filename1,'file')==2
    load(filename1) % So the code can be re-started from the last saved point if it gets interrupted.
else
    loans = zeros(N,N,size(CN,1),Smpl,'uint64');

    for i = 1:size(CN,1)
        C = round(CN(i)*N);
        for s = 1:Smpl
            % STEP 1 - draw block densities
            % See Appendix C for an explanation of this step.
            
            % Set core density
            dens_cT=1;
            
            % Set off-diagonal and periphery densities
            if C>0
                dens_pT = (((DT*N*(N-1))-(C*(C-1))))/(((N-C)*(N-C-1))+(2*C*(N-C)));
            else
                dens_pT = DT;
            end              
            dens_oT = dens_pT;            
            
            % STEP 2
            % Construct blocks using Erdos-Renyi model
            if C>0
                loans(1:C,1:C,i,s) = mrandi(C,C,dens_cT);
                for k = 1:C
                    loans(k,k,i,s) = 0;
                end                
            end
            if C<N
                loans(C+1:N,C+1:N,i,s) = mrandi(N-C,N-C,dens_pT);
                for k = C+1:N
                    loans(k,k,i,s) = 0;
                end                 
            end
            if C>0 && C<N
                loans(1:C,C+1:N,i,s) = mrandi(C,N-C,dens_oT);
                loans(C+1:N,1:C,i,s) = mrandi(N-C,C,dens_oT);
            end

            % STEP 3
            % Ensure true off-diagonal blocks are row/column regular
            if C>0 && C<N
                for h = 1:C
                    if sum(loans(C+1:N,h,i,s))==0
                        loans(C+randi(N-C),h,i,s) = 1;
                    end
                    if sum(loans(h,C+1:N,i,s))==0
                        loans(h,C+randi(N-C),i,s) = 1;
                    end
                end
            end
        end
    end
    clear i k C s dens_pT dens_cT dens_oT h

    % Report average densities of simulated draws
    Dens = zeros(size(CN,1),1);
    Dens_o = zeros(size(CN,1),1);
    Dens_p = zeros(size(CN,1),1);
    Dens_c = zeros(size(CN,1),1);
    for i = 1:size(CN,1)
        C = round(CN(i)*N);
        Dens(i) = sum((sum(sum(loans(:,:,i,:),1),2)./((N^2)-N)),4)/Smpl;
        Dens_c(i) = sum((sum(sum(loans(1:C,1:C,i,:),1),2)./(C*(C-1))),4)/Smpl;
        Dens_o(i) = sum((sum(sum(loans(1:C,C+1:N,i,:),1),2)./(C*(N-C))),4)/Smpl;
        Dens_o(i) = Dens_o(i) + sum((sum(sum(loans(C+1:N,1:C,i,:),1),2)./(C*(N-C))),4)/Smpl;
        Dens_o(i) = Dens_o(i)/2;
        Dens_p(i) = sum((sum(sum(loans(C+1:N,C+1:N,i,:),1),2)./((N-C)*(N-C-1))),4)/Smpl;
        str = ['Dens: ',num2str(Dens(i)),' Core: ',num2str(Dens_c(i)),' Off: ',num2str(Dens_o(i)),' Periph: ',num2str(Dens_p(i))];
        disp(str)        
    end
    clear i C

    CPvec = zeros(N,size(CN,1),Smpl,2);
    NumOpt = zeros(size(CN,1),Smpl,2);
    MinErr = zeros(size(CN,1),Smpl,2);
    SameVec = zeros(size(CN,1),Smpl,2);

    itemp = 1;
    save(filename1,'-v7.3')
end

%% Identify core/periphery from each simulated network
% These temporary variables change with each iteration and are saved. This
% creates a restart point to be used if the code is interrupted.
istart = itemp;

v = 3; % Using the Density-based estimator
for i = istart:size(CN,1)
    for s = 1:Smpl
        [CPvec(:,i,s,1),NumOpt(i,s,1),MinErr(i,s,1),SameVec(i,s,1),~] = IntOptGlob(loans(:,:,i,s),ones(1,N),v,Iter);
        str = ['S: ',num2str(s),'/',num2str(Smpl),' CN: ',num2str(i),'/',num2str(size(CN,1))];
        disp(str)
    end
    itemp = i;
    save(filename1,'-v7.3')
end

v = 5; % Using the Cucuringu et al (2016) estimator
for i = 1:size(CN,1)
    for s = 1:Smpl
        [CPvec(:,i,s,2),NumOpt(i,s,2),MinErr(i,s,2),SameVec(i,s,2),~] = IntOptGlob(loans(:,:,i,s),ones(1,N),v,Iter);
    end
end

%% Define errors relative to the core
% Each incorrect identification is an error. For a maximum error of N.

Err = zeros(size(CN,1),Smpl,2);
Ideal = zeros(N,size(CN,1));
for i = 1:size(CN,1)
    C = round(CN(i)*N);
    if C>0
        Ideal(1:C,i) = 1;
    end
end
clear C

for i = 1:size(CN,1)
    for s = 1:Smpl
        for v = 1:2
            Err(i,s,v) = sum(abs(CPvec(:,i,s,v)-Ideal(:,i)));
        end
    end
end

AvgErr(:,:) = mean(Err,2); % [size(CN,1),Vers]

ErrUB = zeros(size(CN,1),2);
for v = 1:2
    for i = 1:size(CN,1)
        tmp(:,1) = sort(Err(i,:,v),2)';
        ErrUB(i,v) = tmp(round(Smpl*(1-ci)));
        clear tmp
    end
end

%Plot Figure C1
figure
hold all
for v = 1:2
    plot(sum(Ideal,1)',AvgErr(:,v));
end
vnamesUB = vnames;
for v = 1:2
    plot(sum(Ideal,1)',ErrUB(:,v),'--');
    vnamesUB(v,1) = strcat(vnames(v,1),' UB');
end
legend([vnames(1:2); vnamesUB(1:2)]);
title('Average Error - Cuc');
hold off

%% Save output
save(filename2,'-v7.3')
delete(filename1)